/*
 *  IBM eServer eHCA Infiniband device driver for Linux on POWER
 *
 *  Userspace functions
 *
 *  Authors: Khadija Souissi <souissik@de.ibm.com>
 *           Christoph Raisch <raisch@de.ibm.com>
 *
 *  Copyright (c) 2005 IBM Corporation
 *
 *  All rights reserved.
 *
 *  This source code is distributed under a dual license of GPL v2.0 and OpenIB
 *  BSD.
 *
 * OpenIB BSD License
 *
  * OpenIB BSD License
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials
 * provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *  $Id: ehca_umain.c,v 1.14 2006/06/07 10:38:00 nguyen Exp $
 */

#include <infiniband/driver.h>
#include "ehca_uclasses.h"
#include "ehca_utools.h"
#include "ehca_everbs.h"
#include "ipzu_pt_fn.h"


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include <netinet/in.h>

static int cq_assign_qp(struct ehcau_cq *cq, struct ehcau_qp *qp)
{
	unsigned int qp_num = qp->real_qp_num;
	unsigned int key = qp_num & (QP_HASHTAB_LEN-1);
	ehcau_lock(&cq->lockvar);
	LIST_INSERT_HEAD(&cq->qp_hashtab[key],
			 qp, list_entries);
	ehcau_unlock(&cq->lockvar);
	EDEB(7, "cq_num=%x real_qp_num=%x", cq->cq_number, qp_num);
	return 0;
}

static int cq_unassign_qp(struct ehcau_cq *cq, unsigned int real_qp_num)
{
	int ret = -EINVAL;
	unsigned int key = real_qp_num & (QP_HASHTAB_LEN-1);
	struct ehcau_qp *qp = NULL;
	ehcau_lock(&cq->lockvar);
	for (qp = cq->qp_hashtab[key].lh_first;
	     qp!=NULL; qp = qp->list_entries.le_next) {
		if (qp->real_qp_num == real_qp_num) {
			LIST_REMOVE(qp, list_entries);
			EDEB(7, "removed qp from cq .cq_num=%x real_qp_num=%x",
			     cq->cq_number, real_qp_num);
			ret = 0;
			break;
		}
	}
	ehcau_unlock(&cq->lockvar);
	if (ret!=0) {
		EDEB_ERR(4, "qp not found cq_num=%x real_qp_num=%x",
			 cq->cq_number, real_qp_num);
	}
	return ret;
}

struct ehcau_qp* ehca_cq_get_qp(struct ehcau_cq *cq, int real_qp_num)
{
	struct ehcau_qp *ret = NULL;
	unsigned int key = real_qp_num & (QP_HASHTAB_LEN-1);
	struct ehcau_qp *qp = NULL;
	for (qp = cq->qp_hashtab[key].lh_first;
	     qp!=NULL; qp = qp->list_entries.le_next) {
		if (qp->real_qp_num == real_qp_num) {
			ret = qp;
			break;
		}
	}
	return ret;
}

int ehcau_query_device(struct ibv_context *context,
		       struct ibv_device_attr *device_attr)
{
	int ret = 0;
	struct ibv_query_device cmd;
	uint64_t raw_fw_ver;

	EDEB_EN(7, "context=%p", context);

	memset(&cmd, 0, sizeof(cmd));
	ret = ibv_cmd_query_device(context, device_attr, &raw_fw_ver, &cmd,
				   sizeof(cmd));

	if (ret) {
		EDEB_ERR(4, "ibv_cmd_query_device() failed, ret=%x", ret);
	}

	EDEB_EX(7, "context=%p", context);
	return ret;
}

int ehcau_query_port(struct ibv_context *context, uint8_t port,
		     struct ibv_port_attr *attr)
{
	int ret = 0;
	struct ibv_query_port cmd;

	EDEB_EN(7, "context=%p port=%x", context, port);

	memset(&cmd, 0, sizeof(cmd));
	ret = ibv_cmd_query_port(context, port, attr, &cmd, sizeof(cmd));

	if (ret) {
		EDEB_ERR(4,
			 "ibv_cmd_query_port failed ret=%x context=%p port=%x",
			 ret, context, port);
	}

	EDEB_EX(7, "ret=%x context=%p port=%x", ret, context, port);
	return ret;

}

struct ibv_pd *ehcau_alloc_pd(struct ibv_context *context)
{
	struct ibv_pd *pd = NULL;
	struct ibv_alloc_pd cmd;
	struct ibv_alloc_pd_resp resp;
	int ret = 0;

	EDEB_EN(7, "context=%p", context);

	pd = malloc(sizeof(*pd));
	if (pd == NULL) {
		EDEB_ERR(4, "Out of memory to alloc ehcau_pd "
			 "context=%p", context);
		return NULL;
	}

	memset(pd, 0, sizeof(*pd));
	memset(&cmd, 0, sizeof(cmd));
	memset(&resp, 0, sizeof(resp));
	ret = ibv_cmd_alloc_pd(context, pd,
			       &cmd, sizeof(cmd), &resp, sizeof(resp));

	if (ret != 0) {
		EDEB_ERR(4, "ibv_cmd_alloc_pd() failed ret=%x context=%p",
			 ret, context);
		free(pd);
		return NULL;
	}

	EDEB_EX(7, "context=%p ret=ehcau_pd=%p", context, pd);
	return (pd);
}

int ehcau_dealloc_pd(struct ibv_pd *pd)
{
	int ret = 0;

	EHCA_CHECK_ADR(pd);
	EDEB_EN(7, "pd=%p", pd);
	ret = ibv_cmd_dealloc_pd(pd);
	if (ret) {
		EDEB_ERR(4, "ibv_cmd_dealloc_pd failed ret=%x pd=%p", ret, pd);
		return ret;
	}
	free(pd);
	EDEB_EX(7, "pd=%p", pd);
	return (0);
}

struct ibv_cq *ehcau_create_cq(struct ibv_context *context, int cqe,
			       struct ibv_comp_channel *channel, int comp_vector)
{
	struct ibv_create_cq cmd;
	struct ehcau_create_cq_resp resp;
	struct ehcau_cq *my_cq = NULL;
	int ret = 0;
	int i;

	EDEB_EN(7, "context=%p cqe=%x", context, cqe);
	my_cq = malloc(sizeof(*my_cq));
	if (!my_cq) {
		EDEB_ERR(4, "Out of memory context=%p cqe=%x", context, cqe);
		return NULL;
	}

	memset(my_cq, 0, sizeof(*my_cq));
	memset(&cmd, 0, sizeof(cmd));
	memset(&resp, 0, sizeof(resp));
	ret = ibv_cmd_create_cq(context, cqe, channel, comp_vector, &my_cq->ib_cq,
				&cmd, sizeof(cmd),
				&resp.ibv_resp, sizeof(resp));
	if (ret) {
		EDEB_ERR(4, "ibv_cmd_create_cq() failed "
			 "ret=%x context=%p cqe=%x", ret, context, cqe);
		goto create_cq_exit0;
	}

	for (i=0; i<QP_HASHTAB_LEN; i++) {
		LIST_INIT(&my_cq->qp_hashtab[i]);
	}

	/* copy data returned from kernel */
	my_cq->cq_number = resp.cq_number;
	my_cq->token = resp.token;
	/* right most cast is required to avoid gcc warning in 32 bit mode */
	my_cq->ipz_queue.queue = (u8*)(long)resp.ipz_queue.queue;
	my_cq->ipz_queue.current_q_addr = (u8*)(long)resp.ipz_queue.queue;
	my_cq->ipz_queue.qe_size = resp.ipz_queue.qe_size;
	my_cq->ipz_queue.act_nr_of_sg = resp.ipz_queue.act_nr_of_sg;
	my_cq->ipz_queue.queue_length = resp.ipz_queue.queue_length;
	my_cq->ipz_queue.pagesize = resp.ipz_queue.pagesize;
	my_cq->ipz_queue.toggle_state = resp.ipz_queue.toggle_state;
	my_cq->galpas = resp.galpas;

	/* access queue mem to fill page cache */
	memset(my_cq->ipz_queue.queue, 0,
	       my_cq->ipz_queue.queue_length);

	EDEB_EX(7, "ehcau_cq=%p cqn=%x token=%x "
		"ipz_queue.galpa=%p ipz_queue.adr=%p", my_cq,
		my_cq->cq_number, my_cq->token,
		(u64 *) (unsigned long)my_cq->galpas.kernel.fw_handle,
		(u64 *) my_cq->ipz_queue.queue);
	return &my_cq->ib_cq;

      create_cq_exit0:
	EDEB_EX(4, "An error has occured context=%p cqe=%x",
		context, cqe);
	free(my_cq);
	return NULL;
}

int ehcau_destroy_cq(struct ibv_cq *cq)
{
	struct ehcau_cq *my_cq = NULL;
	int cq_num = 0;
	int ret = 0;
	EHCA_CHECK_ADR(cq);
	my_cq = container_of(cq, struct ehcau_cq, ib_cq);
	cq_num = my_cq->cq_number;
	EDEB_EN(7, "ehcau_cq=%p cq_num=%x", my_cq, cq_num);
	ret = ibv_cmd_destroy_cq(cq);
	if (ret) {
		EDEB_ERR(4, "ibv_cmd_destroy_cq() failed ret=%x "
			 "ehcau_cq=%p cq_num=%x", ret, my_cq, cq_num);
		return ret;
	}
	free(my_cq);
	EDEB_EX(7, "ehcau_cq=%p cq_num=%x", my_cq, cq_num);
	return 0;
}

struct ibv_qp *ehcau_create_qp(struct ibv_pd *pd, struct ibv_qp_init_attr *attr)
{
	int ret = 0;
	struct ehcau_qp *my_qp = NULL;
	struct ibv_create_qp cmd;
	struct ehcau_create_qp_resp resp;
	struct ibv_context *context = NULL;
	int ret2 = 0;

	EHCA_CHECK_ADR_P(pd);
	EHCA_CHECK_ADR_P(pd->context);
	context = pd->context;
	EDEB_EN(7, "pd=%p attr=%p", pd, attr);

	my_qp = malloc(sizeof(*my_qp));
	if (!my_qp) {
		EDEB_ERR(4, "Out of memory to alloc qp pd=%p", pd);
		return NULL;
	}

	memset(my_qp, 0, sizeof(*my_qp));
	memset(&cmd, 0, sizeof(cmd));
	memset(&resp, 0, sizeof(resp));

	ret = ibv_cmd_create_qp(pd, &my_qp->ib_qp, attr,
				&cmd, sizeof(cmd),
				&resp.ibv_resp, sizeof resp);

	if (ret != 0) {
		EDEB_ERR(4, "ibv_cmd_create_qp() failed ret=%x pd=%p",
			 ret, pd);
		goto create_qp_exit0;
	}
	/* copy data returned from kernel */
	my_qp->qp_num = resp.qp_num;
	my_qp->token = resp.token;
	my_qp->qp_type = resp.qp_type;
	my_qp->qkey = resp.qkey;
	my_qp->real_qp_num = resp.real_qp_num;
	/* rqueue properties */
	my_qp->ipz_rqueue.queue = (u8*)(long)resp.ipz_rqueue.queue;
	my_qp->ipz_rqueue.current_q_addr = (u8*)(long)resp.ipz_rqueue.queue;
	my_qp->ipz_rqueue.qe_size = resp.ipz_rqueue.qe_size;
	my_qp->ipz_rqueue.act_nr_of_sg = resp.ipz_rqueue.act_nr_of_sg;
	my_qp->ipz_rqueue.queue_length = resp.ipz_rqueue.queue_length;
	my_qp->ipz_rqueue.pagesize = resp.ipz_rqueue.pagesize;
	my_qp->ipz_rqueue.toggle_state = resp.ipz_rqueue.toggle_state;
	/* squeue properties */
	my_qp->ipz_squeue.queue = (u8*)(long)resp.ipz_squeue.queue;
	my_qp->ipz_squeue.current_q_addr = (u8*)(long)resp.ipz_squeue.queue;
	my_qp->ipz_squeue.qe_size = resp.ipz_squeue.qe_size;
	my_qp->ipz_squeue.act_nr_of_sg = resp.ipz_squeue.act_nr_of_sg;
	my_qp->ipz_squeue.queue_length = resp.ipz_squeue.queue_length;
	my_qp->ipz_squeue.pagesize = resp.ipz_squeue.pagesize;
	my_qp->ipz_squeue.toggle_state = resp.ipz_squeue.toggle_state;
	my_qp->galpas = resp.galpas;

	/* access queue mem to fill page cache */
	memset(my_qp->ipz_squeue.queue, 0,
	       my_qp->ipz_squeue.queue_length);
	memset(my_qp->ipz_rqueue.queue, 0,
	       my_qp->ipz_rqueue.queue_length);

	if (attr->send_cq!=NULL) {
		struct ehcau_cq *cq = container_of(attr->send_cq, struct ehcau_cq, ib_cq);
		ret = cq_assign_qp(cq, my_qp);
		if (ret !=0) {
			EDEB_ERR(4, "Couldn't assign qp to send_cq ret=%x", ret);
			goto create_qp_exit1;
		}
		my_qp->send_cq = cq;
	}

	EDEB_EX(7, "ehcau_qp=%p "
		"ipz_queue.galpa=%p ipz_rqueue.adr=%p ipz_squeue.adr=%p",
		my_qp,
		(u64 *) (unsigned long)my_qp->galpas.kernel.fw_handle,
		(u64 *) my_qp->ipz_rqueue.queue,
		(u64 *) my_qp->ipz_squeue.queue);

	return &my_qp->ib_qp;

 create_qp_exit1:
	ret2 = ibv_cmd_destroy_qp(&my_qp->ib_qp);
	if (ret2) {
		EDEB_ERR(4, "ibv_cmd_destroy_qp() failed ret=%x "
			 "qp=%p qp_num=%x", ret2, my_qp, my_qp->qp_num);
	}
 create_qp_exit0:
	EDEB_EX(4, "An error has occured pd=%p", pd);
	free(my_qp);
	return NULL;
}

int ehcau_modify_qp(struct ibv_qp *qp, struct ibv_qp_attr *attr,
		    enum ibv_qp_attr_mask attr_mask)
{
	int ret = 0;
	struct ibv_modify_qp cmd;
	struct ehcau_qp *my_qp = NULL;
	struct ehca_wqe *wqe = NULL;
	int sq_locked = 0;

	EHCA_CHECK_ADR(qp);
	EHCA_CHECK_ADR(attr);
	if (attr_mask == 0) {	/* nothing to modify */
		return ret;
	}
	my_qp = container_of(qp, struct ehcau_qp, ib_qp);

	EDEB_EN(7, "qp=%p qp_num=%x attr=%p attr_mask=%x",
		qp, my_qp->qp_num, attr, attr_mask);

	if ((attr_mask & IBV_QP_STATE)!=0 && attr->qp_state==IBV_QPS_RTS) {
		unsigned int qp_type = -1;
		qp_type = my_qp->qp_type;
		if (IBV_QPT_UD == qp_type) {
			/* lock send queue */
			ehcau_lock(&my_qp->lockvar_s);
			sq_locked = 1;
			/* mark next free wqe */
			wqe=(struct ehca_wqe*)
				my_qp->ipz_squeue.current_q_addr;
			wqe->optype = wqe->wqef = 0xff;
			EDEB(7, "qp_num=%x next_free_wqe=%p",
			     my_qp->qp_num, wqe);
		}
	}

	ret = ibv_cmd_modify_qp(qp, attr, attr_mask, &cmd, sizeof(cmd));
	if (ret) {
		EDEB_ERR(4, "ibv_cmd_modify_qp() failed ret=%x "
			 "qp=%p qp_num=%x", ret, qp, my_qp->qp_num);
	} else if (attr_mask & IBV_QP_STATE) {
		switch (attr->qp_state) {
		case IBV_QPS_RESET:
			/* reset s/r queue pointers */
			ipzu_qeit_reset(&my_qp->ipz_rqueue);
			ipzu_qeit_reset(&my_qp->ipz_squeue);
			break;
		case IBV_QPS_RTS:
			if (sq_locked && wqe!=NULL) {
				my_qp->sqerr_purgeflag = ~wqe->wqef;
			}
			break;
		default: /* nothing to do */
			break;
		}
	}

	if (sq_locked) { /* unlock send queue */
		ehcau_unlock(&my_qp->lockvar_s);
	}

	if (attr_mask & IBV_QP_QKEY) {
		my_qp->qkey = attr->qkey;
	}

	EDEB_EX(7, "ret=%x qp=%p qp_num=%x", ret, qp, my_qp->qp_num);
	return ret;

}

int ehcau_destroy_qp(struct ibv_qp *qp)
{
	int ret = 0;
	struct ehcau_qp *my_qp = NULL;
	u32 qp_num = 0;

	EHCA_CHECK_ADR(qp);
	my_qp = container_of(qp, struct ehcau_qp, ib_qp);
	qp_num = my_qp->qp_num;

	EDEB_EN(7, "qp=%p qp_num=%x", qp, qp_num);

	if (my_qp->send_cq!=NULL) {
		ret = cq_unassign_qp(my_qp->send_cq,
				     my_qp->real_qp_num);
		if (ret !=0) {
			EDEB_ERR(4, "Couldn't unassign qp from send_cq "
				 "ret=%x real_qp_num=%x cq_num=%x",
				 ret, my_qp->real_qp_num,
				 my_qp->send_cq->cq_number);
			goto destroy_qp_exit0;
		}
	}
	ret = ibv_cmd_destroy_qp(qp);
	if (ret) {
		EDEB_ERR(4, "ibv_cmd_destroy_qp() failed ret=%x "
			 "qp=%p qp_num=%x", ret, qp, qp_num);
	} else {
		free(my_qp);
	}

 destroy_qp_exit0:
	EDEB_EX(7, "ret=%x qp=%p qp_num=%x", ret, qp, qp_num);
	return ret;
}

struct ibv_ah *ehcau_create_ah(struct ibv_pd *pd, struct ibv_ah_attr *attr)
{
	struct ehcau_av *my_av = NULL;

	EHCA_CHECK_ADR_P(pd);
	EHCA_CHECK_ADR_P(attr);

	EDEB_EN(7, "pd=%p attr=%p", pd, attr);

	my_av = malloc(sizeof *my_av);
	if (!my_av) {
		EDEB_ERR(4, "no address handle");
		return (NULL);
	}

	memset(my_av, 0, sizeof(*my_av));
	my_av->av.sl = attr->sl;
	my_av->av.dlid = ntohs(attr->dlid);
	my_av->av.slid_path_bits = attr->src_path_bits;
	/* TODO: this needs to be converted as done in kernel space
	   my_av->av.ipd = attr->static_rate;
	*/
	my_av->av.ipd = 0; /* currently just set to max */
	my_av->av.lnh = attr->is_global;
	my_av->av.grh.word_0 |= EHCA_BMASK_SET(GRH_IPVERSION_MASK, 6);
	my_av->av.grh.word_0 |= EHCA_BMASK_SET(GRH_TCLASS_MASK,
					       attr->grh.traffic_class);
	my_av->av.grh.word_0 |= EHCA_BMASK_SET(GRH_FLOWLABEL_MASK,
					       attr->grh.flow_label);
	my_av->av.grh.word_0 |= EHCA_BMASK_SET(GRH_HOPLIMIT_MASK,
					       attr->grh.hop_limit);
	my_av->av.grh.word_0 |= EHCA_BMASK_SET(GRH_NEXTHEADER_MASK, 0x1B);
	/* IB transport */
	my_av->av.grh.word_0 = be64_to_cpu(my_av->av.grh.word_0);

	/* set sgid in grh.word_1 */
	if (attr->is_global != 0) {
		struct ibv_context *context = pd->context;
		struct ibv_port_attr port_attr;
		union ibv_gid gid;
		int rc = 0;
		memset(&port_attr, 0, sizeof(port_attr));
		rc = ibv_query_port(context, attr->port_num, &port_attr);
		if (rc != 0) { /* port number invalid */
			EDEB_ERR(4, "ibv_query_port() failed "
				 "rc=%x context=%p port_num=%x",
				 rc, context, attr->port_num);
			free(my_av);
			return NULL;
		}
		memset(&gid, 0, sizeof(gid));
		rc = ibv_query_gid(context,
				   attr->port_num,
				   attr->grh.sgid_index, &gid);
		if (rc != 0) {
			EDEB_ERR(4, "ibv_query_gid() failed "
				 "rc=%x context=%p port_num=%x "
				 "sgid_index=%x",
				 rc, context, attr->port_num,
				 attr->grh.sgid_index);
			free(my_av);
			return NULL;
		}
		memcpy(&my_av->av.grh.word_1, &gid, sizeof(gid));
	}

	/* see also ehca_av.c
	 * For the time beeing we use a hard coded PMTU of 2048 Bytes.
	 */
	my_av->av.pmtu = 4;	/* TODO */

	/* dgid comes in grh.word_3 */
	memcpy(&my_av->av.grh.word_3, &attr->grh.dgid, sizeof(attr->grh.dgid));

	EDEB_EX(7, "pd=%p attr=%p my_av=%p", pd, attr, my_av);

	return (&my_av->ib_ah);
}

int ehcau_destroy_ah(struct ibv_ah *ah)
{
	EHCA_CHECK_ADR(ah);

	EDEB_EN(7, "ah=%p", ah);
	free(ah);
	EDEB_EX(7, "ah=%p", ah);

	return 0;
}

int ehcau_attach_mcast(struct ibv_qp *qp, union ibv_gid *gid, uint16_t lid)
{
	int ret = 0;
	struct ehcau_qp *my_qp = NULL;
	u32 qp_num = 0;

	EHCA_CHECK_ADR(qp);
	EHCA_CHECK_ADR(gid);
	my_qp = container_of(qp, struct ehcau_qp, ib_qp);

	EHCA_CHECK_QP(my_qp);

	qp_num = my_qp->qp_num;

	EDEB_EN(7, "qp=%p qp_num=%x", qp, qp_num);

	ret = ibv_cmd_attach_mcast(qp, gid, lid);

	if (ret) {

		EDEB_ERR(4, "ehcau_qp=%p qp_num=%x "
			 "ibv_cmd_attach_mcast() failed "
			 "ret=%x", my_qp, my_qp->qp_num, ret);
	}

	EDEB_EX(7, "qp=%p qp_num=%x ret=%x", qp, qp_num, ret);

	return ret;
}

int ehcau_detach_mcast(struct ibv_qp *qp, union ibv_gid *gid, uint16_t lid)
{
	int ret = 0;
	struct ehcau_qp *my_qp = NULL;
	u32 qp_num = 0;

	EHCA_CHECK_ADR(qp);
	EHCA_CHECK_ADR(gid);
	my_qp = container_of(qp, struct ehcau_qp, ib_qp);

	EHCA_CHECK_QP(my_qp);

	qp_num = my_qp->qp_num;

	EDEB_EN(7, "qp=%p qp_num=%x", qp, qp_num);

	ret = ibv_cmd_detach_mcast(qp, gid, lid);

	if (ret) {

		EDEB_ERR(4, "ehcau_qp=%p qp_num=%x "
			 "ibv_cmd_detach_mcast() failed "
			 "ret=%x", my_qp, my_qp->qp_num, ret);
	}

	EDEB_EX(7, "qp=%p qp_num=%x ret=%x", qp, qp_num, ret);

	return ret;
}


int ehcau_query_qp(struct ibv_qp *qp, struct ibv_qp_attr *qp_attr,
		   enum ibv_qp_attr_mask attr_mask, struct ibv_qp_init_attr* init_attr)
{
	int ret = 0;
	struct ehcau_qp *my_qp = NULL;
	struct ibv_query_qp cmd;

	EHCA_CHECK_ADR(qp);
	EHCA_CHECK_ADR(qp_attr);

	my_qp = container_of(qp, struct ehcau_qp, ib_qp);

	EDEB_EN(7, "qp=%p qp_num=%x qp_attr=%p", qp, my_qp->qp_num, qp_attr);

        ret =  ibv_cmd_query_qp(qp, qp_attr, attr_mask, init_attr,
				&cmd, sizeof(cmd));
	if (ret) {
		EDEB_ERR(4, "ehcau_qp=%p qp_num=%x "
			 "ibv_cmd_query_() failed "
			 "ret=%x", my_qp, my_qp->qp_num, ret);
	}

	EDEB_EX(7, "ret=%x qp=%p qp_num=%x", ret, qp, my_qp->qp_num);
	return ret;
}


int ehcau_query_qp_da(struct ibv_qp *qp, struct ehcau_qp_attr_da *qp_attr)
{
	int ret = 0;
	struct ehcau_qp *my_qp = NULL;

	EHCA_CHECK_ADR(qp);
	EHCA_CHECK_ADR(qp_attr);
	my_qp = container_of(qp, struct ehcau_qp, ib_qp);

	EDEB_EN(7, "qp=%p qp_num=%x qp_attr=%p", qp, my_qp->qp_num, qp_attr);

	qp_attr->send_wqe_size = my_qp->ipz_squeue.qe_size;
	qp_attr->send_queue_length = my_qp->ipz_squeue.queue_length;
	qp_attr->send_queue_ptr = my_qp->ipz_squeue.queue;
	qp_attr->recv_wqe_size = my_qp->ipz_rqueue.qe_size;
	qp_attr->recv_queue_length = my_qp->ipz_rqueue.queue_length;
	qp_attr->recv_queue_ptr = my_qp->ipz_rqueue.queue;

	EDEB_EX(7, "ret=%x qp=%p qp_num=%x "
		"send_wqe_size=%x send_queue_size=%lx send_queue_ptr=%p "
		"recv_wqe_size=%x recv_queue_size=%lx recv_queue_ptr=%p",
		ret, qp, my_qp->qp_num,
		qp_attr->send_wqe_size,
		(unsigned long)qp_attr->send_queue_length,
		qp_attr->send_queue_ptr,
		qp_attr->recv_wqe_size,
		(unsigned long)qp_attr->recv_queue_length,
		qp_attr->recv_queue_ptr);
	return ret;
}

/* eof ehca_umain.c */
